/*
 * Decompiled with CFR 0.152.
 */
package dan200.computercraft.core.filesystem;

import com.google.common.base.Splitter;
import com.google.common.io.ByteStreams;
import dan200.computercraft.api.filesystem.Mount;
import dan200.computercraft.api.filesystem.MountConstants;
import dan200.computercraft.api.filesystem.WritableMount;
import dan200.computercraft.core.CoreConfig;
import dan200.computercraft.core.filesystem.FileSystemException;
import dan200.computercraft.core.filesystem.FileSystemWrapper;
import dan200.computercraft.core.filesystem.MountWrapper;
import dan200.computercraft.core.util.IoUtil;
import java.io.Closeable;
import java.io.IOException;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.OpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.Set;
import java.util.regex.Pattern;

public class FileSystem {
    private static final int MAX_COPY_DEPTH = 128;
    private final Map<String, MountWrapper> mounts = new HashMap<String, MountWrapper>();
    private final HashMap<WeakReference<FileSystemWrapper<?>>, SeekableByteChannel> openFiles = new HashMap();
    private final ReferenceQueue<FileSystemWrapper<?>> openFileQueue = new ReferenceQueue();
    private static final Pattern threeDotsPattern = Pattern.compile("^\\.{3,}$");
    private static final char[] specialChars = new char[]{'\"', '*', ':', '<', '>', '?', '|'};
    private static final char[] specialCharsAllowWildcards = new char[]{'\"', ':', '<', '>', '|'};

    public FileSystem(String rootLabel, Mount rootMount) throws FileSystemException {
        this.mount(rootLabel, "", rootMount);
    }

    public FileSystem(String rootLabel, WritableMount rootMount) throws FileSystemException {
        this.mountWritable(rootLabel, "", rootMount);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        HashMap<WeakReference<FileSystemWrapper<?>>, SeekableByteChannel> hashMap = this.openFiles;
        synchronized (hashMap) {
            for (Closeable closeable : this.openFiles.values()) {
                IoUtil.closeQuietly(closeable);
            }
            this.openFiles.clear();
            while (this.openFileQueue.poll() != null) {
            }
        }
    }

    public synchronized void mount(String label, String location, Mount mount) throws FileSystemException {
        Objects.requireNonNull(mount, "mount cannot be null");
        location = FileSystem.sanitizePath(location);
        if (location.contains("..")) {
            throw new FileSystemException("Cannot mount below the root");
        }
        this.mount(new MountWrapper(label, location, mount));
    }

    public synchronized void mountWritable(String label, String location, WritableMount mount) throws FileSystemException {
        Objects.requireNonNull(mount, "mount cannot be null");
        location = FileSystem.sanitizePath(location);
        if (location.contains("..")) {
            throw new FileSystemException("Cannot mount below the root");
        }
        this.mount(new MountWrapper(label, location, mount));
    }

    private synchronized void mount(MountWrapper wrapper) {
        String location = wrapper.getLocation();
        this.mounts.remove(location);
        this.mounts.put(location, wrapper);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void unmount(String path) {
        MountWrapper mount = this.mounts.remove(FileSystem.sanitizePath(path));
        if (mount == null) {
            return;
        }
        this.cleanup();
        HashMap<WeakReference<FileSystemWrapper<?>>, SeekableByteChannel> hashMap = this.openFiles;
        synchronized (hashMap) {
            Iterator<WeakReference<FileSystemWrapper<?>>> iterator = this.openFiles.keySet().iterator();
            while (iterator.hasNext()) {
                WeakReference<FileSystemWrapper<?>> reference = iterator.next();
                FileSystemWrapper wrapper = (FileSystemWrapper)reference.get();
                if (wrapper == null || wrapper.mount != mount) continue;
                wrapper.closeExternally();
                iterator.remove();
            }
        }
    }

    public String combine(String path, String childPath) {
        path = FileSystem.sanitizePath(path, true);
        childPath = FileSystem.sanitizePath(childPath, true);
        if (path.isEmpty()) {
            return childPath;
        }
        if (childPath.isEmpty()) {
            return path;
        }
        return FileSystem.sanitizePath(path + "/" + childPath, true);
    }

    public static String getDirectory(String path) {
        if ((path = FileSystem.sanitizePath(path, true)).isEmpty()) {
            return "..";
        }
        int lastSlash = path.lastIndexOf(47);
        if (lastSlash >= 0) {
            return path.substring(0, lastSlash);
        }
        return "";
    }

    public static String getName(String path) {
        if ((path = FileSystem.sanitizePath(path, true)).isEmpty()) {
            return "root";
        }
        int lastSlash = path.lastIndexOf(47);
        return lastSlash >= 0 ? path.substring(lastSlash + 1) : path;
    }

    public synchronized long getSize(String path) throws FileSystemException {
        return this.getMount(FileSystem.sanitizePath(path)).getSize(FileSystem.sanitizePath(path));
    }

    public synchronized BasicFileAttributes getAttributes(String path) throws FileSystemException {
        return this.getMount(FileSystem.sanitizePath(path)).getAttributes(FileSystem.sanitizePath(path));
    }

    public synchronized String[] list(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        ArrayList<String> list = new ArrayList<String>();
        mount.list(path, list);
        for (MountWrapper otherMount : this.mounts.values()) {
            if (!FileSystem.getDirectory(otherMount.getLocation()).equals(path)) continue;
            list.add(FileSystem.getName(otherMount.getLocation()));
        }
        Object[] array = new String[list.size()];
        list.toArray(array);
        Arrays.sort(array);
        return array;
    }

    public synchronized boolean exists(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        return mount.exists(path);
    }

    public synchronized boolean isDir(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        return mount.isDirectory(path);
    }

    public synchronized boolean isReadOnly(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        return mount.isReadOnly(path);
    }

    public synchronized String getMountLabel(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        return mount.getLabel();
    }

    public synchronized void makeDir(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        mount.makeDirectory(path);
    }

    public synchronized void delete(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        mount.delete(path);
    }

    public synchronized void move(String sourcePath, String destPath) throws FileSystemException {
        sourcePath = FileSystem.sanitizePath(sourcePath);
        destPath = FileSystem.sanitizePath(destPath);
        if (this.isReadOnly(sourcePath) || this.isReadOnly(destPath)) {
            throw new FileSystemException("Access denied");
        }
        if (!this.exists(sourcePath)) {
            throw new FileSystemException("No such file");
        }
        if (this.exists(destPath)) {
            throw new FileSystemException("File exists");
        }
        if (FileSystem.contains(sourcePath, destPath)) {
            throw new FileSystemException("Can't move a directory inside itself");
        }
        MountWrapper mount = this.getMount(sourcePath);
        if (mount == this.getMount(destPath)) {
            mount.rename(sourcePath, destPath);
        } else {
            this.copy(sourcePath, destPath);
            this.delete(sourcePath);
        }
    }

    public synchronized void copy(String sourcePath, String destPath) throws FileSystemException {
        sourcePath = FileSystem.sanitizePath(sourcePath);
        if (this.isReadOnly(destPath = FileSystem.sanitizePath(destPath))) {
            throw new FileSystemException("/" + destPath + ": Access denied");
        }
        if (!this.exists(sourcePath)) {
            throw new FileSystemException("/" + sourcePath + ": No such file");
        }
        if (this.exists(destPath)) {
            throw new FileSystemException("/" + destPath + ": File exists");
        }
        if (FileSystem.contains(sourcePath, destPath)) {
            throw new FileSystemException("/" + sourcePath + ": Can't copy a directory inside itself");
        }
        this.copyRecursive(sourcePath, this.getMount(sourcePath), destPath, this.getMount(destPath), 0);
    }

    private synchronized void copyRecursive(String sourcePath, MountWrapper sourceMount, String destinationPath, MountWrapper destinationMount, int depth) throws FileSystemException {
        if (!sourceMount.exists(sourcePath)) {
            return;
        }
        if (depth >= 128) {
            throw new FileSystemException("Too many directories to copy");
        }
        if (sourceMount.isDirectory(sourcePath)) {
            destinationMount.makeDirectory(destinationPath);
            ArrayList<String> sourceChildren = new ArrayList<String>();
            sourceMount.list(sourcePath, sourceChildren);
            for (String child : sourceChildren) {
                this.copyRecursive(this.combine(sourcePath, child), sourceMount, this.combine(destinationPath, child), destinationMount, depth + 1);
            }
        } else {
            try (SeekableByteChannel source = sourceMount.openForRead(sourcePath);
                 SeekableByteChannel destination = destinationMount.openForWrite(destinationPath, MountConstants.WRITE_OPTIONS);){
                ByteStreams.copy((ReadableByteChannel)source, (WritableByteChannel)destination);
            }
            catch (AccessDeniedException e) {
                throw new FileSystemException("Access denied");
            }
            catch (IOException e) {
                throw FileSystemException.of(e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cleanup() {
        HashMap<WeakReference<FileSystemWrapper<?>>, SeekableByteChannel> hashMap = this.openFiles;
        synchronized (hashMap) {
            Reference<FileSystemWrapper<?>> ref;
            while ((ref = this.openFileQueue.poll()) != null) {
                IoUtil.closeQuietly(this.openFiles.remove(ref));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized FileSystemWrapper<SeekableByteChannel> openFile(MountWrapper mount, SeekableByteChannel channel) throws FileSystemException {
        HashMap<WeakReference<FileSystemWrapper<?>>, SeekableByteChannel> hashMap = this.openFiles;
        synchronized (hashMap) {
            if (CoreConfig.maximumFilesOpen > 0 && this.openFiles.size() >= CoreConfig.maximumFilesOpen) {
                IoUtil.closeQuietly(channel);
                throw new FileSystemException("Too many files already open");
            }
            FileSystemWrapper<SeekableByteChannel> fsWrapper = new FileSystemWrapper<SeekableByteChannel>(this, mount, channel, this.openFileQueue);
            this.openFiles.put(fsWrapper.self, channel);
            return fsWrapper;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeFile(FileSystemWrapper<?> handle) {
        HashMap<WeakReference<FileSystemWrapper<?>>, SeekableByteChannel> hashMap = this.openFiles;
        synchronized (hashMap) {
            this.openFiles.remove(handle.self);
        }
    }

    public synchronized FileSystemWrapper<SeekableByteChannel> openForRead(String path) throws FileSystemException {
        this.cleanup();
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        SeekableByteChannel channel = mount.openForRead(path);
        return this.openFile(mount, channel);
    }

    public synchronized FileSystemWrapper<SeekableByteChannel> openForWrite(String path, Set<OpenOption> options) throws FileSystemException {
        this.cleanup();
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        SeekableByteChannel channel = mount.openForWrite(path, options);
        return this.openFile(mount, channel);
    }

    public synchronized long getFreeSpace(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        return mount.getFreeSpace();
    }

    public synchronized OptionalLong getCapacity(String path) throws FileSystemException {
        path = FileSystem.sanitizePath(path);
        MountWrapper mount = this.getMount(path);
        return mount.getCapacity();
    }

    private synchronized MountWrapper getMount(String path) throws FileSystemException {
        Iterator<MountWrapper> it = this.mounts.values().iterator();
        MountWrapper match = null;
        int matchLength = 999;
        while (it.hasNext()) {
            MountWrapper mount = it.next();
            if (!FileSystem.contains(mount.getLocation(), path)) continue;
            int len = FileSystem.toLocal(path, mount.getLocation()).length();
            if (match != null && len >= matchLength) continue;
            match = mount;
            matchLength = len;
        }
        if (match == null) {
            throw new FileSystemException("/" + path + ": Invalid Path");
        }
        return match;
    }

    private static String sanitizePath(String path) {
        return FileSystem.sanitizePath(path, false);
    }

    public static String sanitizePath(String path, boolean allowWildcards) {
        path = path.replace('\\', '/');
        StringBuilder cleanName = new StringBuilder();
        char[] allowedChars = allowWildcards ? specialCharsAllowWildcards : specialChars;
        for (int i = 0; i < path.length(); ++i) {
            char c = path.charAt(i);
            if (c < ' ' || Arrays.binarySearch(allowedChars, c) >= 0) continue;
            cleanName.append(c);
        }
        path = cleanName.toString();
        ArrayDeque<String> outputParts = new ArrayDeque<String>();
        for (String fullPart : Splitter.on((char)'/').split((CharSequence)path)) {
            String part = fullPart.strip();
            if (part.isEmpty() || part.equals(".") || threeDotsPattern.matcher(part).matches()) continue;
            if (part.equals("..")) {
                if (!outputParts.isEmpty()) {
                    String top = (String)outputParts.peekLast();
                    if (!top.equals("..")) {
                        outputParts.removeLast();
                        continue;
                    }
                    outputParts.addLast("..");
                    continue;
                }
                outputParts.addLast("..");
                continue;
            }
            if (part.length() >= 255) {
                outputParts.addLast(part.substring(0, 255).strip());
                continue;
            }
            outputParts.addLast(part);
        }
        return String.join((CharSequence)"/", outputParts);
    }

    public static boolean contains(String pathA, String pathB) {
        pathA = FileSystem.sanitizePath(pathA).toLowerCase(Locale.ROOT);
        if ((pathB = FileSystem.sanitizePath(pathB).toLowerCase(Locale.ROOT)).equals("..")) {
            return false;
        }
        if (pathB.startsWith("../")) {
            return false;
        }
        if (pathB.equals(pathA)) {
            return true;
        }
        if (pathA.isEmpty()) {
            return true;
        }
        return pathB.startsWith(pathA + "/");
    }

    public static String toLocal(String path, String location) {
        path = FileSystem.sanitizePath(path);
        location = FileSystem.sanitizePath(location);
        assert (FileSystem.contains(location, path));
        String local = path.substring(location.length());
        if (local.startsWith("/")) {
            return local.substring(1);
        }
        return local;
    }
}

